/**
 *  BBall Serversent plugin
 *  Code by YB
 *  Credit to Chu
 *
 *  v1 - 2015-07-23
 *
 *
 *
 *  This plugin replaces the intels with a basketball. Score points by carrying the ball
 *  and bringing it into the enemy team's capture zone (typically should be a basket)
 *
 *  The BBall mode is triggered on room start, when a map whose name starts with "bball"
 *  is started.
 *
 *  A BBall map should contain a Red Intelligence Base, 2 capture zones, a Red Koth
 *  Control Point, and a Blue Koth Control Point.
 *
 *
 *
 *
 *  This plugin is required, although it shouldn't break clients who connect without
 *  downloading the plugin.
 *
 *
 *  When a Character grabs (contacts with) a Ball, the Ball disappears (is destroyed)
      and the Character's Player becomes the carrier.
 *  When a Character dies, if its Player was the carrier, a new Ball is spawned and its
      team is set to the other team. Momentum is conserved.
 *  When a Ball is spawned, it respawns after BALL_RESPAWN_TIME seconds.
 *  Respawning a Ball means spawning it with no team on IntelligenceBaseRed.
 *  When a Ball is given a team, it is reset to neutral after TEAM_RESET_TIME seconds.
 *  When a Character scores (contacts with a KothControlPoint whose team is not his),
      a Ball respawns and the Character dies without spawning a new Ball.
 *  A Player can drop the Ball if its Character is a carrier. A new Ball is spawned and
      its team is set to the same team. Momentum is conserved.
 *
 *
 *  Server updates:
 *      Ball spawn
 *      Ball score
 *      Ball reset team
 *      Ball respawn
 *      Ball + carrier update
 *    
 *
 *  Client inputs:
 *      Drop ball. Input in PlayerControl, Do in BallController, Send in BallNetworker
 *
 *  TODO
 *      Test Networking
 *      Test Functions
 *          Scoring
 *      Test map resetting
 *      Test Syncing
 */

globalvar bball;
bball = id;


//-------------------------------------------
//  Constants
//-------------------------------------------

BALL_RESPAWN_TIME   = 10;  /* How long until the ball respawns after being dropped, in seconds */
TEAM_RESET_TIME     = 0.5; /* How long until the ball is reset to neutral team, in seconds */
DROP_TIME           = 0.5; /* How long until a player can grab the ball after dropping it, in seconds */

// USER EVENTS FOR BALL
BALL_GRAB           = ev_user0;
BALL_DROP           = ev_user1;
BALL_RESPAWN        = ev_user2;
BALL_KILL           = ev_user3;
BALL_SCORE          = ev_user4;

// USER EVENTS FOR PLAYER INPUT
PLAYER_INPUT_DROP   = ev_user0;

// USER EVENTS FOR NETWORKING
NET_SEND_DROP       = ev_user0;
NET_SEND_GRAB       = ev_user1;
NET_SEND_SCORE      = ev_user2;
NET_SEND_RESETTEAM  = ev_user3;
NET_SEND_RESPAWN    = ev_user4;
NET_SEND_UPDATE     = ev_user11;
NET_READ_UPDATE     = ev_user12; /* Client side, read server messages */
NET_READ_INPUTS     = ev_user13; /* Server side, read client inputs */


// NETWORK BYTES
//    From Server
NET_SRV_UPDATE      = 0;
NET_SRV_SCORE       = 1;
NET_SRV_DROP        = 2;
NET_SRV_RESETTEAM   = 3;
NET_SRV_RESPAWN     = 4;
NET_SRV_GRAB        = 5;

NET_NOCARRIER       = 254;

//    From Client
NET_CLT_DROP        = 0;



//-------------------------------------------
//  Ball Hud
//-------------------------------------------
BallHud = object_add();
object_set_parent(BallHud, CTFHUD);
object_set_depth(BallHud, object_get_depth(CTFHUD));

BallHudSprite = sprite_add(directory + "\hud.png", 1, true, false, 0, 0);
object_set_sprite(BallHud, BallHudSprite);
sprite_set_offset(BallHudSprite, sprite_get_width(BallHudSprite) / 2, sprite_get_yoffset(CTFHUD));

BallHudBallSprite = sprite_add(directory + "\hudBall.png", 3, true, false, 0, 0);
sprite_set_offset(BallHudBallSprite, sprite_get_width(BallHudBallSprite) / 2, sprite_get_height(BallHudBallSprite) / 2);

BallHudTimerSprite = sprite_add(directory + "\hudTimer.png", 8, true, false, 0, 0);
sprite_set_offset(BallHudTimerSprite, sprite_get_width(BallHudTimerSprite) / 2, sprite_get_height(BallHudTimerSprite) / 2);


object_event_add(BallHud, ev_step, ev_step_normal, '
    var redIsCapping, blueIsCapping;
    var ball;
    ball = noone;
    with(bball.Ball) {
        ball = id;
    }
    if(ball == noone) exit;

    redIsCapping = ball.team == TEAM_RED;
    blueIsCapping = ball.team == TEAM_BLUE;

    if (timer > 0)
        timer -= 1 * global.delta_factor;
    if ((global.redCaps >= global.caplimit) or (global.blueCaps >= global.caplimit) or timer<=0) {
        if (global.redCaps > global.blueCaps)
            global.winners = TEAM_RED;
        else if (global.blueCaps > global.redCaps)
            global.winners = TEAM_BLUE;
        else if (!redIsCapping and !blueIsCapping and global.blueCaps == global.redCaps)
            global.winners = TEAM_SPECTATOR;
    }
      
    if (timer <= 0 and global.winners == -1)
      overtime = true;
');

object_event_add(BallHud, ev_draw, 0, '

    xoffset = view_xview[0];
    yoffset = view_yview[0];
    xsize = view_wview[0];
    ysize = view_hview[0];
    xshift = -320*global.timerPos;
    yshift = 5*global.timerPos;
    var ballDir, ballx, bally, ballstatus;

    var ball;
    ball = noone;
    with(bball.Ball) {
      ball = id;
    }
    if(ball == noone) exit;
    
    if (global.myself == -1)
        exit;
    
    //Drawing intel status and arrows:
    
    ballx = xoffset+xsize/2;
    bally = yoffset+ysize-70;
    
    draw_set_alpha(1);
    
    draw_sprite_ext(sprite_index, 0, xoffset+xsize/2, yoffset+ysize, 3, 3, 0, c_white, 1);
	

    draw_set_color(c_black);
    draw_set_valign(fa_middle);
    draw_set_halign(fa_center);

    //showing the amount of caps, and the amount of caps to win
    draw_text_transformed(xoffset+xsize/2-55, yoffset+ysize-25, string(global.redCaps),2,2,0);
    draw_text_transformed(xoffset+xsize/2+55, yoffset+ysize-25, string(global.blueCaps),2,2,0);
    // this next piece of code helps to resize the value of the numbers depending on the limit as not to overflow
    if (global.caplimit <= 9)
    {
        draw_text_transformed(xoffset+xsize/2, yoffset+ysize-15, string(global.caplimit),2,2,0);
    }
    else
    {
        if (global.caplimit > 999)
        {
            draw_sprite_ext(infinity,0,xoffset+xsize/2-1, yoffset+ysize-17, 2, 2, 0, c_white, 1);
        }
        else
        {
            draw_text(xoffset+xsize/2, yoffset+ysize-15, string(global.caplimit));   
        }
    }
      
    
    // Point to the ball

    ballstatus = 0;
    // If nobodys carrying it
    if (ball.carrier == noone)
    {
        ballDir = point_direction(ballx, bally, ball.x, ball.y);
        if (ball.alarm[bball.ALARM_RESPAWN] <= 0)
            ballstatus = 2; // is home
        else
            ballstatus = 0; // is down
    }

    // Someone s carrying it
    else
    {
        with(ball.carrier)
        {
            // If we re carrying it, it should point where we need to go
            if (id == global.myself.object) {

                var goalx, goaly;
                with(CaptureZone) {
                    if(cp.team != global.myself.team) {
                        goalx = cp.x;
                        goaly = cp.y;
                    }
                }

                ballDir = point_direction(ballx, bally, goalx, goaly);
                ballstatus=3; // is taken by self
            } 

            // Otherwise it should point at the person who s carrying it
            else {

                ballDir = point_direction(ballx, bally, x, y);
                ballstatus=1; // is taken
            }
        }
    }
     
	var timerRatio;
	timerRatio = 0;	
	if(ballstatus == 0) {
		timerRatio = 1 - (ball.alarm[bball.ALARM_RESPAWN] / (bball.BALL_RESPAWN_TIME * 30 / global.delta_factor));
    }
	draw_sprite_ext(bball.BallHudTimerSprite, floor(timerRatio * 8), xoffset+xsize/2, yoffset+ysize-69, 3, 3, 0, c_white, 1);
	
	var teamOffset;
	if(ball.team = TEAM_RED)		teamOffset = 1;
	else if(ball.team = TEAM_BLUE)	teamOffset = 2;
	else 							teamOffset = 0;
	draw_sprite_ext(bball.BallHudBallSprite, teamOffset, xoffset+xsize/2, yoffset+ysize-69, 3, 3, 0, c_white, 1);

    
    var teamColor, ballRespawnTime;
    if(ball.team == TEAM_RED) {
        teamColor = make_color_rgb(165, 70, 64);
        teamOffset = 0;
    } else if(ball.team == TEAM_BLUE) {
        teamColor = make_color_rgb(73, 93, 104);
        teamOffset = 1;
    } else {
        if(global.myself.team == TEAM_RED) {
            teamColor = make_color_rgb(165, 70, 64);
            teamOffset = 0;
        } else {
            teamColor = make_color_rgb(73, 93, 104);
            teamOffset = 1;
        }
    }
    ballRespawnTime = bball.BALL_RESPAWN_TIME * 30 / global.delta_factor;

    
    draw_sprite_ext(IntelArrowS,    teamOffset, ballx, bally, 3, 3, ballDir,    c_white, 1);
    draw_sprite_ext(IntelStatusS,   ballstatus, ballx, bally, 2, 2, 0,          c_white, 1);
');







//-------------------------------------------
//  Ball Networker
//-------------------------------------------
BallNetworker = object_add();


object_event_add(BallNetworker, ev_create, 0, '
    frameCount = 0;
    buffer = buffer_create();
    arg0 = noone;
');

object_event_add(BallNetworker, ev_destroy, 0, '
    buffer_destroy(buffer);
');

object_event_add(BallNetworker, ev_step, 0, '
    if(global.isHost) {
        event_perform(ev_other,bball.NET_READ_INPUTS);
        frameCount += global.delta_factor;
        if((frameCount mod 6) == 0) {
            event_perform(ev_other,bball.NET_SEND_UPDATE);
        }
    } else {
        event_perform(ev_other,bball.NET_READ_UPDATE);
    }
');

// Do send
object_event_add(BallNetworker, ev_step, ev_step_end, '
    if(buffer_size(buffer) > 0) {
        PluginPacketSend(bball.packetID, buffer);
        buffer_clear(buffer);
    }
');


object_event_add(BallNetworker, ev_other, NET_SEND_DROP, '
    if(global.isHost) {
        write_ubyte(buffer, bball.NET_SRV_DROP);
        event_perform(ev_other,bball.NET_SEND_UPDATE); // otherwise it desyncs a lot
    } else {
        write_ubyte(buffer, bball.NET_CLT_DROP);
    }
');

/** arg0 : carrier player id */
object_event_add(BallNetworker, ev_other, NET_SEND_GRAB, '
    write_ubyte(buffer, bball.NET_SRV_GRAB);
    write_ubyte(buffer, arg0);
');

/** arg0 : capping team */
object_event_add(BallNetworker, ev_other, NET_SEND_SCORE, '
    write_ubyte(buffer, bball.NET_SRV_SCORE);
    write_ubyte(buffer, arg0);
');

object_event_add(BallNetworker, ev_other, NET_SEND_RESETTEAM, '
    write_ubyte(buffer, bball.NET_SRV_RESETTEAM);
');

object_event_add(BallNetworker, ev_other, NET_SEND_RESPAWN, '
    write_ubyte(buffer, bball.NET_SRV_RESPAWN);
');


object_event_add(BallNetworker, ev_other, NET_SEND_UPDATE, '
    var ballInstance;
    with(bball.Ball) {
        ballInstance = id;
    }
    write_ubyte(buffer, bball.NET_SRV_UPDATE);
    if(ballInstance.carrier == noone) {
        write_ubyte(buffer, bball.NET_NOCARRIER);
        write_ushort(buffer, ballInstance.x);
        write_ushort(buffer, ballInstance.y);
        write_byte(buffer, ballInstance.hspeed * 5);
        write_byte(buffer, ballInstance.vspeed * 5);
        write_ubyte(buffer, ballInstance.team);
		var alarmVal;
		alarmVal = alarm[bball.ALARM_RESPAWN];
		alarmVal *= global.delta_factor / bball.BALL_RESPAWN_TIME;
        write_ubyte(buffer, alarmVal * 255);
    } else {
        var carrierIndex;
        carrierIndex = ds_list_find_index(global.players, ballInstance.carrier.player);
        write_ubyte(buffer, carrierIndex);
    }
');

/** As client, read server updates */
object_event_add(BallNetworker, ev_other, NET_READ_UPDATE, '
    var buf, byteHeader;
    buf = PluginPacketGetBuffer(bball.packetID);
    while(buf != -1) {
        byteHeader = read_ubyte(buf);
        switch(byteHeader) {

            //----------    UPDATE
            //
            case    bball.NET_SRV_UPDATE:

                var ballInstance, carrierIndex;
                ballInstance = noone;
                with(bball.Ball) {
                    ballInstance = id;
                }
                if(ballInstance == noone) {
                    ballInstance = instance_create(0,0,bball.Ball);
                }

                carrierIndex = read_ubyte(buf);
                if(carrierIndex == bball.NET_NOCARRIER) {

                    with(ballInstance) {
                        x = read_ushort(buf);
                        y = read_ushort(buf);
                        hspeed = read_byte(buf) / 5;
                        vspeed = read_byte(buf) / 5;
                        team = read_ubyte(buf);
						var alarmVal;
						alarmVal = read_ubyte(buf) / 255;
						alarmVal *= bball.BALL_RESPAWN_TIME / global.delta_factor;
						alarm[bball.ALARM_RESPAWN] = alarmVal;
						if(carrier != noone) {
							event_perform(ev_other, bball.BALL_DROP);
						}
                    }
                } else {
                    var playerCarrier;
                    playerCarrier = ds_list_find_value(global.players, carrierIndex);
                    with(ballInstance) {
						if(carrier != playerCarrier.object) {
							event_perform(ev_other, bball.BALL_DROP);
							carrier = playerCarrier.object;
							if(playerCarrier.object != -1) {
								event_perform(ev_other, bball.BALL_GRAB);
							}
						}
                    }
                }
                break;

            //----------    EVENTS
            //
            case    bball.NET_SRV_DROP:
				with(bball.Ball) {
					event_perform(ev_other, bball.BALL_DROP);
				}
                break;

            case    bball.NET_SRV_GRAB:
                var carrierIndex, playerCarrier;
                carrierIndex = read_ubyte(buf);
                playerCarrier = ds_list_find_value(global.players, carrierIndex);
				if(playerCarrier.object != -1) {
					with(bball.Ball) {
						carrier = playerCarrier.object;
						event_perform(ev_other, bball.BALL_GRAB);
					}
				}
                break;

            case    bball.NET_SRV_SCORE:
                with(bball.Ball) {
                    capTeam = read_ubyte(buf);
                    event_perform(ev_other, bball.BALL_SCORE);
                }
                break;

            case    bball.NET_SRV_RESETTEAM:
                with(bball.Ball) {
                    team = TEAM_SPECTATOR;
                    alarm[ALARM_TEAM] = 0;
                }
                break;

            case    bball.NET_SRV_RESPAWN:
				with(bball.Ball) {
					event_perform(ev_other, bball.BALL_RESPAWN);
				}
                break;

            default:
                show_error("BBall Serversent Plugin Networking error: unexpected header " + string(byteHeader), false);
        }
		PluginPacketPop(bball.packetID);
		buf = PluginPacketGetBuffer(bball.packetID);
    }
');

/** As server, read client inputs */
object_event_add(BallNetworker, ev_other, NET_READ_INPUTS, '
    var buf, player, byteHeader;
	buf = PluginPacketGetBuffer(bball.packetID);
    while(buf != -1) {
        player = PluginPacketGetPlayer(bball.packetID);
        byteHeader = read_ubyte(buf);
        switch(byteHeader) {
            case    bball.NET_CLT_DROP:
                with(bball.Ball) {
                    if(carrier == player.object) {
                        event_perform(ev_other, bball.BALL_DROP);
                    }
                }
                break;
            default:
                // we re the server, I d rather silently die and drop the packet
                // than throw an error potentially block everything.
        }
		PluginPacketPop(bball.packetID);
		buf = PluginPacketGetBuffer(bball.packetID);
    }
');








//-------------------------------------------
//    Ball Controller
//-------------------------------------------
BallController = object_add();


object_event_add(BallController, ev_create, 0, '
    with(IntelligenceBaseRed) {
        instance_create(x, y, bball.Ball);
    }
    with(CTFHUD) {
        instance_change(bball.BallHud, false);
    }
');

object_event_add(BallController, ev_step, ev_step_end, '
    with(Intelligence) {
        instance_destroy();
    }
    with(IntelligenceBaseBlue) {
        instance_destroy();
    }
    if(!instance_exists(bball.Ball)) {
        with(IntelligenceBaseRed) {
            instance_create(x, y, bball.Ball);
        }
    }
    if(!instance_exists(bball.BallNetworker)) {
        instance_create(0,0,bball.BallNetworker);
    }
    with(ControlPoint) {
        locked = true;
    }
');

object_event_add(BallController, ev_other, PLAYER_INPUT_DROP, '
    with(bball.Ball) {
        if(carrier == global.myself.object) {
            with(bball.BallNetworker) {
                event_perform(ev_other, bball.NET_SEND_DROP);
            }
            if(global.isHost) {
                event_perform(ev_other, bball.BALL_DROP);
            }
        }
    }
');













//-------------------------------------------
//  Ball
//-------------------------------------------
Ball = object_add();
object_set_depth(Ball, object_get_depth(Character) - 1);

BallSprite = sprite_add(directory + "\ball.png", 3, true, false, 0, 0);
object_set_sprite(Ball, BallSprite);
sprite_set_offset(BallSprite, sprite_get_width(BallSprite) / 2, sprite_get_height(BallSprite) / 2);

ALARM_RESPAWN = 0;
ALARM_TEAM    = 1;

object_event_add(Ball, ev_create, 0, '
    carrier             = noone;            /** Character */
    lastCarrierPlayer   = noone;            /** Player */
    team                = TEAM_SPECTATOR;   /** When dropping */
    capTeam             = TEAM_SPECTATOR;   /** When scoring */
');
object_event_add(Ball, ev_step, ev_step_begin, '
	if(!variable_local_exists("carrier")) {	
		carrier             = noone;            /** Character */
		lastCarrierPlayer   = noone;            /** Player */
		team                = TEAM_SPECTATOR;   /** When dropping */
		capTeam             = TEAM_SPECTATOR;   /** When scoring */
	}
');

object_event_add(Ball, ev_alarm, ALARM_RESPAWN, '
    event_perform(ev_other, bball.BALL_RESPAWN);
');

object_event_add(Ball, ev_alarm, ALARM_TEAM, '
    team = TEAM_SPECTATOR;
');


//  Collisions
//
object_event_add(Ball, ev_collision, Character, '
    if(global.isHost)
    if(global.winners == -1)
    if(carrier == noone)
    if(other.bball_dropTime <= 0)
    if(team == TEAM_SPECTATOR or team == other.team) {
        carrier = other.id;
        event_perform(ev_other, bball.BALL_GRAB);
        with(bball.BallNetworker) {
            arg0 = ds_list_find_index(global.players, other.carrier.player);
            event_perform(ev_other, bball.NET_SEND_GRAB);
        }
    }
');

object_event_add(Ball, ev_collision, CaptureZone, '
    if(global.isHost) {
        if(other.cp.team != TEAM_SPECTATOR) {
            capTeam = other.cp.team;
            with(bball.BallNetworker) {
                arg0 = other.capTeam;
                event_perform(ev_other, bball.NET_SEND_SCORE);
            }
            event_perform(ev_other, bball.BALL_SCORE);
        }
    }
');

//  Draw
//
object_event_add(Ball, ev_draw, 0, '
    var teamOffset;
    if(team == TEAM_RED)        teamOffset = 1;
    else if(team == TEAM_BLUE)  teamOffset = 2;
    else                        teamOffset = 0;
    draw_sprite_ext(sprite_index, teamOffset, x, y, 1, 1, 0, c_white, 1);
');

//-------------------------------------------
//    Ball Physics
//-------------------------------------------



// Physics
object_event_add(Ball, ev_step, ev_step_normal, '

    // keep within bounds
    x = max(0, min(map_width(), x));
    if(y < 0) y = 0;
    if(y > map_height()) {
        event_perform(ev_other, bball.BALL_RESPAWN)
        exit;
    }

    if((!instance_exists(carrier)) or (carrier == -1)) {
        carrier = noone;
    }

    // proper physics
    if(carrier != noone) {
        x = carrier.x;
        y = carrier.y;
        hspeed = carrier.hspeed;
        vspeed = carrier.vspeed;
        lastCarrierPlayer = carrier.player;
		team = carrier.player.team;
    } else {

        gunSetSolids();
        with(BulletWall) solid = false;

        speed = min(15, speed);

        if (place_free(x, y+1))
            vspeed += 0.6;
        else
        {
            vspeed = min(vspeed, 0);
            hspeed = hspeed * delta_mult(0.95);
        }
        if (vspeed > 11)
            vspeed = 11;
            
        if (speed < 0.2)
            speed = 0;
        
        if(!place_free(x+hspeed, y+vspeed)) {
            really_move_contact_solid(point_direction(x,y,x+hspeed,y+vspeed), speed);

            if(!place_free(x,y+sign(vspeed)))
            {
                vspeed *= -0.8;
                if(!place_free(x+hspeed,y))
                {
                    really_move_contact_solid(point_direction(x,y,x+hspeed,y+vspeed), speed);
                    hspeed *= -0.7;
                }
            }
            if(!place_free(x+sign(hspeed),y))
            {
                hspeed *= -0.7;
                if(!place_free(x,y+vspeed))
                {
                    really_move_contact_solid(point_direction(x,y,x+hspeed,y+vspeed), speed);
                    vspeed *= -0.8;
                }
            }
        }
        
        x += hspeed * global.delta_factor;
        y += vspeed * global.delta_factor;
        x -= hspeed;
        y -= vspeed;

        gunUnsetSolids();
    }
');

//-------------------------------------------
//    Ball Functions
//-------------------------------------------

object_event_add(Ball, ev_other, BALL_GRAB, '
    if(carrier != noone) {
		carrier.bball_hasBall = true;
		sound_play(IntelGetSnd);
		var isMe;
		isMe = (global.myself == carrier.player);
		recordEventInLog(6, carrier.player.team, carrier.player.name, isMe);
		if (global.myself == carrier.player)
		{
			if !instance_exists(NoticeO)
				instance_create(0,0,NoticeO);
			with (NoticeO)
				notice = NOTICE_HAVEINTEL;
		}
		alarm[bball.ALARM_TEAM] = 0;
		alarm[bball.ALARM_RESPAWN] = 0;
	}
');

object_event_add(Ball, ev_other, BALL_DROP, '
    if(carrier != noone) {
        sound_play(IntelDropSnd);
        var isMe;
        isMe = (global.myself == carrier.player);
        recordEventInLog(5, carrier.player.team, carrier.player.name, isMe); 

        speed *= 1.5;
        carrier.bball_hasBall = false;
        carrier.bball_dropTime = bball.DROP_TIME * 30 / global.delta_factor;
        alarm[bball.ALARM_TEAM] = bball.TEAM_RESET_TIME * 30 / global.delta_factor;
        alarm[bball.ALARM_RESPAWN] = bball.BALL_RESPAWN_TIME * 30 / global.delta_factor;
        carrier = noone;
    
    }
');

object_event_add(Ball, ev_other, BALL_RESPAWN, '
    var spawnX, spawnY;
    with(IntelligenceBaseRed) {
        spawnX = x;
        spawnY = y;
    }
    x = spawnX;
    y = spawnY;
    hspeed = 0;
    vspeed = 0;
    team = TEAM_SPECTATOR;
    capTeam = TEAM_SPECTATOR;
    alarm[bball.ALARM_TEAM] = 0;
    alarm[bball.ALARM_RESPAWN] = 0;
    lastCarrierPlayer = noone;
    if(carrier != noone) {
        carrier.bball_hasBall = false;
        carrier = noone;
    }
');

object_event_add(Ball, ev_other, BALL_KILL, '
    alarm[bball.ALARM_TEAM] = bball.TEAM_RESET_TIME * 30 / global.delta_factor;
    alarm[bball.ALARM_RESPAWN] = bball.BALL_RESPAWN_TIME * 30 / global.delta_factor;
    carrier = noone;
    if(lastCarrierPlayer.team == TEAM_BLUE) {
        team = TEAM_RED;
    } else {
        team = TEAM_BLUE;
    }
    var killer;
    killer = noone;
    with(lastCarrierPlayer) {
        if(variable_local_exists("lastDamageDealer")) {
            killer = lastCarrierPlayer.lastDamageDealer;
        }
    }
    if(killer != noone) {
        killer.stats[DEFENSES] += 1;
        killer.roundStats[DEFENSES] += 1;
        killer.stats[POINTS] += 1;
        killer.roundStats[POINTS] += 1;
        recordEventInLog(4, killer.team, killer.name, global.myself == killer);
    }
    sound_play(IntelDropSnd);
');

/** Requires capTeam to be set */
object_event_add(Ball, ev_other, BALL_SCORE, '
    sound_play(IntelPutSnd);
    var otherTeam; // the team to wish the point is given
    if(capTeam == TEAM_RED)         otherTeam = TEAM_BLUE;
    else if(capTeam == TEAM_BLUE)   otherTeam = TEAM_RED;
    else                            otherTeam = TEAM_SPECTATOR;
    if(lastCarrierPlayer == noone) {
        recordEventInLog(3, otherTeam, "", false);
    } else {
        var isMe;
        isMe = (lastCarrierPlayer == global.myself);
        lastCarrierPlayer.stats[CAPS] += 1;
        lastCarrierPlayer.roundStats[CAPS] += 1;
        if(otherTeam == lastCarrierPlayer.team) {
            lastCarrierPlayer.stats[POINTS] += 2;
            lastCarrierPlayer.roundStats[POINTS] += 2;
            recordEventInLog(3, otherTeam, lastCarrierPlayer.name, isMe);
        } else {
            recordEventInLog(3, otherTeam, lastCarrierPlayer.name + " (og!)", isMe);
        }
    }
	// if youre a client, the host is going to sync later
	if(global.isHost) {
		if(otherTeam == TEAM_RED) {
				global.redCaps += 1;
		} else if(otherTeam == TEAM_BLUE) {
				global.blueCaps += 1;
		}
	}
	var keepCapTeam;
	keepCapTeam = capTeam;
    event_perform(ev_other, bball.BALL_RESPAWN);
	team = keepCapTeam;
    GameServer.syncTimer = 1;
');




//-------------------------------------------
//    Overrides
//-------------------------------------------

//    Make stuff happen
//
object_event_add(PlayerControl, ev_step, ev_step_begin, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
        if(!instance_exists(bball.BallController)) {
            instance_create(0,0,bball.BallController);
        }
    }
');

//    Make Control Points compliant
//
object_event_add(KothRedControlPoint, ev_create, 0, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
        mode = 0;
    }
');
object_event_add(KothRedControlPoint, ev_step, 0, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
        mode = 0;
        capping = 0;
        cappingTeam = -1;
        team = TEAM_RED;
    }
');
object_event_add(KothBlueControlPoint, ev_create, 0, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
        mode = 0;
    }
');
object_event_add(KothBlueControlPoint, ev_step, 0, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
        mode = 0;
        capping = 0;
        cappingTeam = -1;
        team = TEAM_BLUE;
    }
');
object_event_add(CaptureZone, ev_alarm, 0, '
    if(string_lower(string_copy(global.currentMap,1,5)) == "bball") {
		alarm[1] = 2;
    }
');
object_event_add(CaptureZone, ev_alarm, 1, '
	if(!instance_exists(cp)) {
		// oh god I hope this works
		var spawnRed, spawnBlu;
		spawnRed = instance_nearest(x,y,SpawnPointRed);
		spawnBlu = instance_nearest(x,y,SpawnPointBlue);
		if(point_distance(x, y, spawnRed.x, spawnRed.y) < point_distance(x, y, spawnBlu.x, spawnBlu.y)) {
			if(instance_exists(KothRedControlPoint)) {
				cp = instance_nearest(x,y,KothRedControlPoint);
			} else {
				cp = instance_create(x, y, KothRedControlPoint);
			}
		} else {
			if(instance_exists(KothBlueControlPoint)) {
				cp = instance_nearest(x,y,KothBlueControlPoint);
			} else {
				cp = instance_create(x, y, KothBlueControlPoint);
			}
		}
	}
');

//    Player input
//
object_event_add(PlayerControl, ev_step, ev_step_begin, '
    if(keyboard_check_pressed(global.drop)) {
        with(bball.BallController) {
            event_perform(ev_other, bball.PLAYER_INPUT_DROP);
        }
    }
');

//    Character var
//
object_event_add(Character, ev_create, 0, '
    bball_hasBall = false;
    bball_dropTime = 0;
');

//    Character death
//
object_event_add(Character, ev_destroy, 0, '
    if(bball_hasBall) {
        with(bball.Ball) {
            event_perform(ev_other, bball.BALL_KILL);
        }
    }
');

//    Character speed influence
//    Character ball drop
//    Character uber
//
object_event_add(Character, ev_step, ev_step_normal, '
    with(bball.Ball) {
        other.bball_hasBall = (carrier == other.id);
    }
    if(bball_hasBall) {
        if(ubered) {
            with(bball.Ball) {
                event_perform(ev_other, bball.BALL_DROP);
            }
        }
        if(global.winners != -1) {
            with(bball.Ball) {
                event_perform(ev_other, bball.BALL_DROP);
            }
        }
        if(abs(hspeed) > 3) {
            hspeed *= 1-((1 - 0.96) * global.delta_factor);
        }
    }
    // Drop time
    bball_dropTime = max(0, bball_dropTime - global.delta_factor);
');

//    Weapons influence
//
//  Rocket
object_event_add(Rocket, ev_destroy, 0, '
    if(exploded)
    {
        with(bball.Ball)
        {
            if (distance_to_object(other) < other.blastRadius){
                motion_add(point_direction(other.x,other.y,x,y),15-15*(distance_to_object(other)/other.blastRadius));
                rotspeed=random(151)-75;
            }
            lastCarrierPlayer = other.ownerPlayer;
        }
    }
');

//  Mine
object_event_add(Mine, ev_destroy, 0, '
    if(exploded)
    {
        with(bball.Ball)
        {
            if (distance_to_object(other) < other.affectRadius * 0.75)
            {
                motion_add(point_direction(other.x,other.y,x,y),15-15*(distance_to_object(other)/other.affectRadius));
                rotspeed=random(151)-75;
            }
            lastCarrierPlayer = other.ownerPlayer;
        }
    }
');

//  Flamethrower
object_event_add(Flamethrower, ev_other, ev_user2, '
    if(justBlast)
    {
        with(bball.Ball)
        {
            dir = point_direction(other.x, other.y, x, y);
            dist = point_distance(other.x, other.y, x, y);
            angle = abs((dir-other.owner.aimDirection)+720) mod 360;
            if collision_circle(x,y,25,other.poof,false,true)
            {
                motion_add(other.owner.aimDirection, other.blastStrength*(1-dist/other.blastDistance) );
            }
            lastCarrierPlayer = other.ownerPlayer;
        }
    }
');